/*
 * Driver for the Analog Devices digital potentiometers (I2C bus)
 *
 * Copyright (C) 2010 Michael Hennerich, Analog Devices Inc.
 *
 * Licensed under the GPL-2 or later.
 */

#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/platform_device.h>

#include "ad525x_dpot.h"
#include <linux/backlight.h>
#include <linux/ad_backlight.h>

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */
#include <mach/gpio.h>


#define PROC_LCD_FILE   "lcdBklt"
#define PROCFS_MAX_SIZE     2048

#define MAX_BRIGHTNESS 100 // max value is 100%
#define MIN_BRIGHTNESS 0

#define MAX_DEV_BRIGHTNESS 127 //device MAX value,

struct ad_bl_data {
	struct i2c_client *i2cclient;
};

struct lcdbacklight {
	u32 brightness;
	u32 backlightLevelOnBatteryDef;
	u32 backlightLevelOnExtDef;
};

static struct lcdbacklight *gLcdBacklight;
static struct backlight_device *gBlData;
static int initializeOff = 0;
/* ------------------------------------------------------------------------- */
/* I2C bus functions */
static int write_d8(void *client, u8 val)
{
	return i2c_smbus_write_byte(client, val);
}

static int write_r8d8(void *client, u8 reg, u8 val)
{
	return i2c_smbus_write_byte_data(client, reg, val);
}

static int write_r8d16(void *client, u8 reg, u16 val)
{
	return i2c_smbus_write_word_data(client, reg, val);
}

static int read_d8(void *client)
{
	return i2c_smbus_read_byte(client);
}

static int read_r8d8(void *client, u8 reg)
{
	return i2c_smbus_read_byte_data(client, reg);
}

static int read_r8d16(void *client, u8 reg)
{
	return i2c_smbus_read_word_data(client, reg);
}

static const struct ad_dpot_bus_ops bops = {
	.read_d8	= read_d8,
	.read_r8d8	= read_r8d8,
	.read_r8d16	= read_r8d16,
	.write_d8	= write_d8,
	.write_r8d8	= write_r8d8,
	.write_r8d16	= write_r8d16,
};

static int ad_backlight_update_status(struct backlight_device *bl)
{
	int brightness = bl->props.brightness;
	int hw_brightness;
	int max = bl->props.max_brightness;
	char buffer [50];
	int n;
	struct ad_bl_data *pb = dev_get_drvdata(&bl->dev);

	hw_brightness = brightness;
	if (bl->props.invert_brightness) {
		hw_brightness = max - brightness;
	}
	//dev_err(&bl->dev, "----- brightness = %3i hw_brightness = %3i\n", brightness, hw_brightness);
	/* Need to set the gpio to truly enable/disable the backlight. */

	n=sprintf (buffer, "%d", hw_brightness);

	crest_set_rdac0(&pb->i2cclient->dev,buffer,n);

	return 0;
}

static int ad_backlight_get_brightness(struct backlight_device *bl)
{
	return bl->props.brightness;
}

static int ad_backlight_check_fb(struct backlight_device *bl, struct fb_info *info)
{
	return 0;
}


/* 0-255 is passed into this function */
static u32 calcDeviceToUI(int brtness)
{
	/* convert 127 to 100, multiplied times 10 to avoid floating point*/
	u32 val;
	if (brtness) //avoid divide by zero
		val = ((brtness*100) / MAX_DEV_BRIGHTNESS); //max is 127 converted to 100%
	else
		val=0;

	return (val);
}
/* 0-100% is passed in and percentage is returned */
static u32 calcUiToDevice(int brtness)
{
	/* convert 100 to 127 */
	u32 val;
	if (brtness)  //avoid divide by zero
		val = ((brtness * MAX_DEV_BRIGHTNESS)/100); // max is 100% converted to 127
	else
		val=0;

	return (val);
}

/**
 * The structure keeping information about the /proc file
 *
 */
static struct proc_dir_entry *ad_file;

/**
 * The buffer (2k) for this module
 *
 */
static char procfs_buffer_ad[PROCFS_MAX_SIZE];

/**
 * The size of the data held in the buffer
 *
 */
static unsigned long procfs_buffer_size_ad = 0;

/**
 * This funtion is called when the /proc file is read
 *
 */
static ssize_t procfs_read_ad(struct file *filep,      /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{

   char * p = procfs_buffer_ad;
   static int finished = 0;

   /* needed to stop from continuously printing */
   if ( finished == 1 ) { finished=0; return 0; }
   finished = 1;

   p += sprintf ( p, "BRIGHTNESS=%d\n" , gLcdBacklight->brightness );
   p += sprintf ( p, "BACKLIGHT_LEVEL_ON_BATTERY_DEF=%d\n", gLcdBacklight->backlightLevelOnBatteryDef );
   p += sprintf ( p, "BACKLIGHT_LEVEL_ON_EXT_DEF=%d\n", gLcdBacklight->backlightLevelOnExtDef );

   procfs_buffer_size_ad = p - procfs_buffer_ad;

   if ( copy_to_user(buffer, procfs_buffer_ad, procfs_buffer_size_ad) ) {
              return -EFAULT;
   }

  return procfs_buffer_size_ad;
}

static ssize_t procfs_write_ad(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;
	int tempVal=0;

	procfs_buffer_size_ad = len;
	if (procfs_buffer_size_ad > PROCFS_MAX_SIZE ) {
		procfs_buffer_size_ad = PROCFS_MAX_SIZE;
	}

	if ( copy_from_user(procfs_buffer_ad, buffer, procfs_buffer_size_ad) )
	{
		return -EFAULT;
	}
	tag = strsep ( tempPtr, "=" );
	if ( strcmp ( tag, "BRIGHTNESS" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &tempVal );
		if (tempVal <= MAX_BRIGHTNESS)
		{
			gLcdBacklight->brightness=tempVal;
			gBlData->props.brightness=calcUiToDevice(gLcdBacklight->brightness);
			ad_backlight_update_status(gBlData);
		}
		else
		{
			printk(KERN_ERR "brightness percentage value is out of range\n");
		}
	}
	if ( strcmp ( tag, "BACKLIGHT_LEVEL_ON_BATTERY_DEF" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &tempVal );
		if (tempVal <= MAX_BRIGHTNESS)
		{
			gLcdBacklight->backlightLevelOnBatteryDef=tempVal;
		}
		else
		{
			printk(KERN_ERR "backlightLevelOnBatteryDef percentage value is out of range\n");
		}
	}
	if ( strcmp ( tag, "BACKLIGHT_LEVEL_ON_EXT_DEF" ) == 0 )
	{
		value = strsep ( tempPtr, "=" );
		sscanf ( value, "%d", &tempVal );
		if (tempVal <= MAX_BRIGHTNESS)
		{
			gLcdBacklight->backlightLevelOnExtDef=tempVal;
		}
		else
		{
			printk(KERN_ERR "backlightLevelOnExtDef percentage value is out of range\n");
		}
	}
	return procfs_buffer_size_ad;

}

static int module_permission_ad(struct inode *inode, int op, unsigned int foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int procfs_open_ad(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int procfs_close_ad(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_ad_File = {
        .read    = procfs_read_ad,
        .write   = procfs_write_ad,
        .open    = procfs_open_ad,
        .release = procfs_close_ad,
};

static struct inode_operations Inode_Ops_ad_File = {
        .permission = module_permission_ad,    /* check for permissions */
};

static int suspend(struct backlight_device *bl, struct ad_bl_data *pb)
{
	suspendPin(bl->props.enable);
	suspendPin(bl->props.enable_signals);
	suspendPin(bl->props.enable_power);
	return 0;
}

static int ad_backlight_suspend(struct i2c_client *client,
				 pm_message_t state)
{
	struct backlight_device *bl = gBlData;
	struct ad_bl_data *pb = dev_get_drvdata(&bl->dev);

	suspend(bl,pb);
	return 0;
}

static int resume(struct backlight_device *bl, struct ad_bl_data *pb)
{
	resumePin(bl->props.enable_power);
	resumePin(bl->props.enable_signals);
	resumePin(bl->props.enable);
	return 0;
}

static int ad_backlight_resume(struct i2c_client *client)
{
	struct backlight_device *bl = gBlData;
	struct ad_bl_data *pb = dev_get_drvdata(&bl->dev);

	resume(bl,pb);
	return 0;
}


static int __devinit ad_dpot_i2c_probe(struct i2c_client *client,
				      const struct i2c_device_id *id)
{
	struct ad_dpot_bus_data bdata = {
		.client = client,
		.bops = &bops,
	};

	struct ad_dpot_id dpot_id = {
		.name = (char *) &id->name,
		.devid = id->driver_data,
	};

	if (!i2c_check_functionality(client->adapter,
				     I2C_FUNC_SMBUS_WORD_DATA)) {
		dev_err(&client->dev, "SMBUS Word Data not Supported\n");
		return -EIO;
	}

	return ad_dpot_probe(&client->dev, &bdata, &dpot_id);
}

static ssize_t ad_state_show(struct device *dev,
			      struct device_attribute *attr, char *buf)
{
//	struct backlight_device *bl = gBlData;
//	struct ad_bl_data *pb = dev_get_drvdata(&bl->dev);

	return sprintf(buf, "err: not implemented yet\n");
}

static ssize_t ad_state_store(struct device *dev,
			      struct device_attribute *attr,
			      const char *buf, size_t count)
{
	struct backlight_device *bl = gBlData;
	struct ad_bl_data *pb = dev_get_drvdata(&bl->dev);

	if(strstr(buf, "suspend") != NULL)
		suspend(bl,pb);
	else if(strstr(buf, "resume") != NULL)
		resume(bl,pb);

	return count;
}
static DEVICE_ATTR(state, 644, ad_state_show, ad_state_store);

static const struct backlight_ops ad_backlight_ops = {
	.update_status	= ad_backlight_update_status,
	.get_brightness	= ad_backlight_get_brightness,
	.check_fb = ad_backlight_check_fb,
};

static int __devinit crestad_dpot_i2c_probe(struct i2c_client *client,
				      const struct i2c_device_id *id)
{
	struct backlight_properties props;
	struct platform_ad_backlight_data *data = client->dev.platform_data;
	struct backlight_device *bl;
	struct ad_bl_data *pb;
	int ret;

	pb = kzalloc(sizeof(*pb), GFP_KERNEL);
	if (!pb) {
		dev_err(&client->dev, "no memory for state\n");
		return -ENOMEM;
	}
	pb->i2cclient = client;

	memset(&props, 0, sizeof(struct backlight_properties));
	props.initialize_off = initializeOff;

	props.type = BACKLIGHT_RAW;
	props.max_brightness = data->max_brightness;
	props.invert_brightness = data->invert_brightness;
	props.enable = data->enable;
	props.enable_power = data->enable_power;
	props.enable_signals = data->enable_signals;

	if(props.enable.gpio_pin)
		gpio_request(props.enable.gpio_pin, "disp1-bklt_en_3v3");
	if(props.enable_power.gpio_pin)
		gpio_request(props.enable_power.gpio_pin, "vled_en_3v3");
	if(props.enable_signals.gpio_pin)
		gpio_request(props.enable_signals.gpio_pin, "vled_sig_en");

	bl = backlight_device_register(dev_name(&client->dev), &client->dev, pb,
					       &ad_backlight_ops, &props);
	gBlData=bl;

	gLcdBacklight = kzalloc(sizeof(*gLcdBacklight), GFP_KERNEL);
	if (!gLcdBacklight) {
		return -ENOMEM;
	}

	gLcdBacklight->brightness = calcDeviceToUI(bl->props.brightness);
	gLcdBacklight->backlightLevelOnBatteryDef = gLcdBacklight->brightness;
	gLcdBacklight->backlightLevelOnExtDef = gLcdBacklight->brightness;

	if(!props.initialize_off)
		resume(bl,pb);

	/* create the /proc file */
	ad_file = create_proc_entry(PROC_LCD_FILE, 0644, NULL);
	/* check if the /proc file was created successfuly */
	if (ad_file == NULL){
		printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
			   PROC_LCD_FILE);
		return -ENOMEM;
	}
	else
	{
		//ad_file->owner = THIS_MODULE;
		ad_file->proc_iops = &Inode_Ops_ad_File;
		ad_file->proc_fops = &File_Ops_ad_File;
		ad_file->mode = S_IFREG | S_IRUGO | S_IWUSR;
		ad_file->uid = 0;
		ad_file->gid = 0;
		ad_file->size = 80;
	}

	ret = device_create_file(&bl->dev, &dev_attr_state);
	if (ret)
		dev_err(&bl->dev, "Error %d on creating file\n", ret);


	return ad_dpot_i2c_probe(client, id);
}

static int __devexit ad_dpot_i2c_remove(struct i2c_client *client)
{
	return ad_dpot_remove(&client->dev);
}

static const struct i2c_device_id ad_dpot_id[] = {
	{"ad5258", AD5258_ID},
	{"ad5259", AD5259_ID},
	{"ad5251", AD5251_ID},
	{"ad5252", AD5252_ID},
	{"ad5253", AD5253_ID},
	{"ad5254", AD5254_ID},
	{"ad5255", AD5255_ID},
	{"ad5241", AD5241_ID},
	{"ad5242", AD5242_ID},
	{"ad5243", AD5243_ID},
	{"ad5245", AD5245_ID},
	{"ad5246", AD5246_ID},
	{"ad5247", AD5247_ID},
	{"ad5248", AD5248_ID},
	{"ad5280", AD5280_ID},
	{"ad5282", AD5282_ID},
	{"adn2860", ADN2860_ID},
	{"ad5273", AD5273_ID},
	{"ad5171", AD5171_ID},
	{"ad5170", AD5170_ID},
	{"ad5172", AD5172_ID},
	{"ad5173", AD5173_ID},
	{"ad5272", AD5272_ID},
	{"ad5274", AD5274_ID},
	{}
};
MODULE_DEVICE_TABLE(i2c, ad_dpot_id);

static struct i2c_driver ad_dpot_i2c_driver = {
	.driver = {
		.name	= "ad_dpot",
		.owner	= THIS_MODULE,
	},
	.probe		= ad_dpot_i2c_probe,
	.remove		= __devexit_p(ad_dpot_i2c_remove),
	.id_table	= ad_dpot_id,
};

static const struct i2c_device_id ad_dpot_crestid[] = {
	{"crestad5247", AD5247_ID},
	{}
};
MODULE_DEVICE_TABLE(i2c, ad_dpot_crestid);

static struct i2c_driver crestad_dpot_i2c_driver = {
	.driver = {
		.name	= "crestad_dpot",
		.owner	= THIS_MODULE,
	},
	.probe		= crestad_dpot_i2c_probe,
	.remove		= __devexit_p(ad_dpot_i2c_remove),
	.suspend	= ad_backlight_suspend,
	.resume		= ad_backlight_resume,
	.id_table	= ad_dpot_crestid,
};

static int __init ad_dpot_i2c_init(void)
{
	return i2c_add_driver(&crestad_dpot_i2c_driver);
//	return i2c_add_driver(&ad_dpot_i2c_driver);
}
module_init(ad_dpot_i2c_init);

static void __exit ad_dpot_i2c_exit(void)
{
	i2c_del_driver(&ad_dpot_i2c_driver);
}
module_exit(ad_dpot_i2c_exit);

/**
 *	ad5x_setup - process command line options
 *	@options: string of options
 *
 *	Process command line options for backlight subsystem.
 *
 *	NOTE: This function is a __setup and __init function.
 *            It only stores the options.
 *
 */
static int __init ad_setup(char *options)
{
	sscanf(options,"%d",&(initializeOff));
	return 1;
}
__setup("initOff=", ad_setup);

MODULE_AUTHOR("Michael Hennerich <hennerich@blackfin.uclinux.org>");
MODULE_DESCRIPTION("digital potentiometer I2C bus driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("i2c:ad_dpot");
